r"""
Colorbar
========

The :meth:`pygmt.Figure.colorbar` method creates a color scalebar.
The colormap is set via the ``cmap`` parameter. A full list of available
color palette tables can be found at :gmt-docs:`reference/cpts.html`.
Use the ``frame`` parameter to add labels to the **x** and **y** axes
of the colorbar by appending **+l** followed by the desired text. To add
and adjust the annotations (**a**) and ticks (**f**) append the letter
followed by the desired interval. The placement of the colorbar is set
via the ``position`` parameter. There are the following options:

- **j/J**: placed inside/outside the plot bounding box using a
  :doc:`2-character justification code </techref/justification_codes>`, e.g.,
  ``position="jTR"`` for Top Right.
- **g**: using map coordinates, e.g. ``position="g170/-45"`` for longitude
  170° East, latitude 45° South.
- **x**: using paper coordinates, e.g. ``position="x5c/7c"`` for 5 cm, 7 cm
  from anchor point.
- **n**: using normalized (0-1) coordinates, e.g. ``position="n0.4/0.8"``.

Note that the anchor point defaults to Bottom Left (**BL**). Append ``+h`` to
``position`` to get a horizontal colorbar instead of a vertical one (``+v``).
"""

# %%
import pygmt

fig = pygmt.Figure()
fig.basemap(region=[0, 3, 6, 9], projection="x3c", frame=["af", "WSne+tColorbars"])

# ============
# Create a colorbar designed for seismic tomography - roma
# Colorbar is placed at Bottom Center (BC) by default if no position is given
# Add quantity and unit as labels ("+l") to the x and y axes
# Add annotations ("+a") in steps of 0.5 and ticks ("+f") in steps of 0.1
fig.colorbar(cmap="SCM/roma", frame=["xa0.5f0.1+lVelocity", "y+lm/s"])

# ============
# Create a colorbar showing the scientific rainbow - batlow
fig.colorbar(
    cmap="SCM/batlow",
    # Colorbar positioned at map coordinates (g) longitude/latitude 0.3/8.7,
    # with a length/width (+w) of 4 cm by 0.5 cm, and plotted horizontally (+h)
    position="g0.3/8.7+w4c/0.5c+h",
    box=True,
    frame=["x+lTemperature", "y+l°C"],
    scale=100,
)

# ============
# Create a colorbar suitable for surface topography - oleron
fig.colorbar(
    cmap="SCM/oleron",
    # Colorbar placed outside the plot bounding box (J) at Middle Right (MR),
    # offset (+o) by 1 cm horizontally and 0 cm vertically from anchor point,
    # with a length/width (+w) of 7 cm by 0.5 cm and a box for NaN values (+n)
    position="JMR+o1c/0c+w7c/0.5c+n+mc",
    # Note that the label 'Elevation' is moved to the opposite side and plotted
    # vertically as a column of text using '+mc' in the position parameter
    # above
    frame=["x+lElevation", "y+lm"],
    scale=10,
)

# ============
# Create a colorbar suitable for categorical data - hawaii
# Set up the colormap
pygmt.makecpt(
    cmap="SCM/hawaii",
    series=[0, 3, 1],
    # Comma-separated string for the annotations of the colorbar
    color_model="+cA,B,C,D",
)
# Plot the colorbar
fig.colorbar(
    cmap=True,  # Use colormap set up above
    # Colorbar placed inside the plot bounding box (j) in the Bottom Left (BL) corner,
    # with an offset (+o) by 0.5 cm horizontally and 0.8 cm vertically from the anchor
    # point, and plotted horizontally (+h)
    position="jBL+o0.5c/0.8c+h",
    box=True,
    # Divide colorbar into equal-sized rectangles
    equalsize=0.5,
)

fig.show()
